\ This is a heavily commented Polymath source code file.  The backslash
\ (\) character means that whatever follows is comment and will not be
\ interpreted.
\
\ This application file will demonstrate the following:
\
\    -  Local variables
\    -  Local words
\    -  Internal words
\    -  Both RPN (reverse polish notation) and Algebraic expressions
\    -  Redefining words
\    -  Documenting words
\    -  Forward seeking words
\
\ This file provides two different applications:
\
\    -  Integration of a supplied function using either
\       the trapezoidal rule or Simpson's rule.
\
\    -  Solve for zeroes of a function.

: DoTrapezoid { lo hi n } {{ f }}      \ LO HI N are local variables
                                       \ which grab values off the data
                                       \ stack.
                                       \    Lo - Lower bound
                                       \    Hi - Upper bound
                                       \    n  - Number of integration
                                       \         points

                                       \ "f" is a word local to this
                                       \ definition.  At run time it
                                       \ will be replaced with the next
                                       \ word to follow in the input
                                       \ stream.  For example, with
                                       \      DoTrapezoid sin
                                       \ "f" will be replaced with "sin"

  (hi-lo)/(n-1)    {  h  }             \ Local variables may be defined
                                       \ anywhere within a definition.
                                       \ As usual, they will grab from
                                       \ the stack whatever you have
                                       \ placed there.

                                       \ Also note the use of algebraic
                                       \ notation. Polymath understands
                                       \ both algebraice notation (with
                                       \ no intervening spaces) and
                                       \ reverse polish notation (with
                                       \ intervening spaces).  This
                                       \ expression could have been
                                       \ written as:
                                       \
                                       \       hi lo - n 1 - /

  (f(lo)+f(hi))/2  { sum }             \ "sum" is initialized with this
                                       \ method to half the sum of the
                                       \ end points

  hi-(h/2) --> hi                      \ The value of the upper bound is
                                       \ decremented by a half step to
                                       \ avoid any problems with round-
                                       \ off on the final comparison.

  begin break                          \ Start of an indefinite
                                       \ begin-while-repeat loop.

                                       \ "break" is for safety.  If a
                                       \ solution can not be found and
                                       \ this routine seems to be in an
                                       \ infinite loop, then a simple
                                       \ press of a key will get you out
                                       \ of trouble.

     Lo+h --> Lo                       \ A local variable (Lo) is used
                                       \ later in the definition just
                                       \ as you would any other
                                       \ variable.

  while( lo<hi )                       \ This indefinite loop sums up
                                       \ all values of the function at
                                       \ increments of h
     Sum +f(lo) --> sum
  repeat

  h*sum; Invisible                     \ The final step - multiply by
                                       \ the length of the integration
                                       \ step and leave the result on
                                       \ the stack.

: odd { x } oor(x,1);                  \ Used by the next word, this
                                       \ ensures that the number of
                                       \ integration steps is odd.

: DoSimpson { lo hi n }  {{ f }}       \ Integration with Simpson's
                                       \ method

  odd(n) --> n                         \ The number of integration steps
                                       \ must be an odd number.
  (hi-lo)/(n-1) { h }
  h+h           { h2 }

  (f(lo)+(4*f(lo+h))+f(hi))   { sum }  \ Initialize sum

  hi-(h/2) --> hi                      \ The value of the upper bound is
                                       \ decremented by a half step to
                                       \ avoid any problems with round-
                                       \ off on the final comparison.

  begin break                          \ With Simpson's method, values
     Lo+h2 --> Lo                      \ of the function are multiplied
  while( lo<hi )                       \ by alternating factors of 2 & 4
     Sum                               \ These are then accumulated into
     (2*f(lo)) +                       \ "sum"
     (4*f(lo+h)) + --> sum
  repeat

  h*sum/3;                             \ The final step - multiply by
                                       \ the length of the integration
                                       \ step and leave the result on
                                       \ the stack.

  invisible                            \ The effect of INVISIBLE is to
                                       \ inhibit this word from being
                                       \ displayed with WORDS.  This
                                       \ keeps the word from view, but
                                       \ still needs to be available in
                                       \ the dictionary later when it is
                                       \ used within double quotes.

Defer Method                           \ Define a new word METHOD with a
Invisible                              \ null definition.  It will be
                                       \ given a definition in the next
                                       \ operation.

Make Method DoSimpson                  \ METHOD is given the definition
                                       \ of DOSIMPSON.  Later on, using
                                       \ METHOD will be equivalent to
                                       \ using DOSIMPSON unless, of
                                       \ course, it is redefined in the
                                       \ same way.

: simp make("method dosimpson");       \ The function of SIMP and TRAP
: trap make("method dotrapezoid");     \ is simply to provide convenient
                                       \ commands for redefining the
                                       \ method of integration.

\ Note how MAKE is being used here. Above, when it was executed (rather
\ than compiled), it grabbed the next two words METHOD & DOSIMPSON.
\ It could be compiled into a new definition, but the new word would
\ then have the same effect.  The double quotes are used to bring the
\ next words into the definition.  The effect of double quotes is
\ similar to single quotes but, instead of placing their contents onto
\ the alpha stack, they shove their contents onto the front end of the
\ command line.

: Integrate method ;                   \ The definition of INTEGRATE
                                       \ is now very simple.  It
                                       \ simply executes METHOD which
                                       \ may be defined to do either
                                       \ the trapezoid rule or
                                       \ Simpson's rule.

\ ----------------------------------------------------------------------

\ The following application solves for zeroes of a function.  The
\ function is supplied as a word by the user.  It must always have the
\ behaviour that it consumes one value from the stack, and it returns a
\ value to the stack.

defer speed                            \ This is a word which will have
invisible                              \ a variable meaning.  It will be
                                       \ incorporated later into SOLVE
                                       \ provide a number of different
                                       \ display options.

: goslow key drop ; invisible          \ First display option.
                                       \ After each iteration, the
                                       \ current values will be
                                       \ displayed, and execution will
                                       \ stop, until a key is pressed.

: gofast ; invisible                   \ Current values will still be
                                       \ displayed, but there will be no
                                       \ pause.

: slow make("speed goslow");           \ A word to set slow speed
: fast make("speed gofast");           \ A word to set fast speed

Fast                                   \ Have this word execute now, to
                                       \ initialize fast SPEED

: field string 20 justify ;            \ A convenient word to help
                                       \ formatting the current values
                                       \ on the command line display.

 0.1 real variable delta               \ This "global" variable will be
                                       \ used later to get two starting
                                       \ values from a single initial
                                       \ guess.

: SetDel --> delta ;                   \ This is a convenient word for a
                                       \ function key assignment.

: solve { guess } {{ f }}              \ GUESS grabs an initial guess
                                       \ from the stack.  F will be
                                       \ replaced by the next word to
                                       \ follow in the input stream.

  guess-delta   { x1 }                 \ SOLVE iterates toward a
  guess+delta   { x2 }                 \ solution by linear
                                       \ interpolation between two
                                       \ points on the curve.  Two
                                       \ initial values are obtained
                                       \ from the single initial guess.

  f(x1) { y1 }                         \ The local variable Y1 is
                                       \ created and initialized.

  begin                                \ Start of an indefinite BEGIN-
                                       \ WHILE-REPEAT loop.

     f(x2,break) { y2 }                \ The local variable Y2 is
                                       \ created and initialized.

     view(field(x2)&field(y2))         \ The current iteration values
                                       \ are formatted and displayed on
                                       \ the command line without
                                       \ pausing the execution.

     speed                             \ The adjustable-meaning word
                                       \ discussed earlier.  It  will
                                       \ either do nothing and let
                                       \ execution proceed, or else it
                                       \ will stop and wait for a key to
                                       \ be pressed.

  while(                               \ This is the part of the loop
                                       \ which checks for termination.

     (round(y2)<>0)                    \ ROUND sets the number on the
                                       \ stack to the value as it is
                                       \ displayed.  This provides the
                                       \ means for controlling the
                                       \ accuracy of the solution.  If
                                       \ you want the solution
                                       \ accurate to two places, for
                                       \ example, simply type FIX(2) to
                                       \ set the display format for two
                                       \ places, before running SOLVE.

     (y1<>y2)                          \ This is for safety when you are
                                       \ asking for the maximum
                                       \ accuracy and the solution
                                       \ can not quite reach zero
                                       \ because of round off problems.

     and)                              \ If either of the above
                                       \ comparisons are FALSE, then a
                                       \ solution has been reached, and
                                       \ this loop is exited.

     x2                                \ Place the current best solution
                                       \ on the stack, in preparation
                                       \ for being placed in X1 as the
                                       \ "old" value.

     ((x1*y2)-(x2*y1))/(y2-y1)         \ Calculate a new improved value
                                       \ by linear interpolation.

     --> x2 --> x1                     \ Update the current values for
     y2 --> y1                         \ the two points on the curve.

  Repeat                               \ This is the end of the loop.
                                       \ Go back to "begin"

  X2;                                  \ The solution has been found.
                                       \ Now place it on the stack for
                                       \ the benefit of the user and
                                       \ quit.

Hide Odd                               \ These are a couple of words
Hide Field                             \ which were required elsewhere
                                       \ in the application, but can now
                                       \ be removed from the dictionary.

\ ----------------------------------------------------------------------

\ This application is, for all practical purposes, complete.  It can
\ be tested out and used.  What follows is extra.  After the application
\ has been written, the next step is to write some on-line
\ documentation.  This is the same on-line documentation that is already
\ used with the existing Polymath words.

Document Simp
Simp   ( --- )

Set the method to be used by INTEGRATE, which will then integrate by
Simpson's rule.  With this method, progressive values of the function
being integrated are multiplied by the following factors:

                        1 4 2 4 ... 4 2 4 1 $

Document Trap
Trap   ( --- )

Set the method to be used by INTEGRATE, which will then integrate by
the trapezoidal rule.  With this method, progressive values of the
function being integrated are multiplied by the following factors:

                        .5 1 1 1 ... 1 1 1 .5$

Document Integrate
Integrate  ( x1 x2 n --- y ) <function>

Integrate a function.

The function is a word which is provided by the user.  It may be an
already defined word or a new word.  The only requirement is that it
takes a single value from the stack and returns a single value to the
stack.

X1 and X2 are the lower and upper bounds for integration, respectively.
N is the number of points at which the function will be evaluated.  When
integrating by Simpson's rule, N must be an odd number, but it will be
incremented by one, if necessary.

This word will integrate either by the trapezoidal or by Simpson's rule.
The method to be used may be set with the words TRAP and SIMP.
.bre
In the following example, the SIN function is integrated from 0 to pi.
The SIN function will be evaluated at 51 points and will be integrated
with both available methods:


          0 pi 21  Trap Integrate Sin    \ result = 1.99588597
          0 pi 21  Simp Integrate Sin    \ result = 2.00000678

Also see the words~TRAP and~SIMP for further information.$

\ There are a couple of notes to be made here.  This last word
\ documentation makes use of a message board directive (.bre).  It is
\ identified by the period (.) in the first column.  This particular
\ directive forces a page or screen break.

\ Also note the use of a tilde (~) character.  When displayed on the
\ message board the tilde is not displayed, but the word which
\ immediately follows is highlighted.  A cursor may be moved across the
\ highlighted words.  When selected, the word is incorporated into a
\ command, the default command being to explain or display the on-line
\ documentation for the selected word.

Document Slow
Slow  ( --- )

Set the display of the iteration steps for~SOLVE   When the current
values are displayed, execution will stop.  When a key is pressed,
execution will continue.$

Document Fast
Fast  ( --- )

Set the display of the iteration steps for~SOLVE   When the current
values are displayed, execution will continue.$

Document SetDel
SetDel  ( delta --- )

When you run~Solve your initial guess will be expanded into two initial
values, by adding the value of delta to your guess and by subtracting
the value of delta from your guess.

Delta is initially set to 0.1.  It is not likely that you will need to
change this value for most cases.$

Document Solve
Solve  ( guess --- ) <function>

Solve for zeroes of a function.  GUESS is the point where you want to
start searching.  The FUNCTION is supplied as a word by the user.  It
must always have the behaviour that it consumes one value from the
stack, and it returns a value to the stack.

If there is no possible solution, continue searching forever or until it
overflows.  Success or lack of success should be evident from the
ongoing display.  To stop the search, simply press any key.
.bre
In the following example, a function F is first created which will
square the number on the stack and then subtract the value of the
variable Z.  The correct solution is then the square root of the value
currently in Z (two possible solutions).

          Real variable z
          : f dup * z - ;

And now to solve for both zeroes:

          10 Solve F
        --10 Solve F

For additional information, also see the words: ~FAST~SLOW &~SETDEL $

\ This application should be reasonably "friendly" at this point.  You
\ can list the words in the dictionary ("Polymath Words").  You would
\ then see the few words which run the application.  Documentation may
\ then be displayed for any of these words (e.g. "Explain Solve").

\ There is still more that can be done, however, to make the application
\ even friendlier.  This application will be saved later as a PACKAGE.
\ This means that it only needs to be compiled once.  When the already
\ compiled version of it is loaded, the word STARTUP will automatically
\ be executed, if it has been defined.

\ STARTUP provides a useful place to assign the functions of the
\ application to function keys.  Before defining this word, let's
\ define some related words which will be useful to also assign to
\ function keys:


: Intg                                 \ This word is a version of
                                       \ INTEGRATE which will prompt for
                                       \ the necessary input.  This is
                                       \ the version which will get a
                                       \ function key assignment.

  Message('Enter X1 X2 & N.')          \ MESSAGE sends text from the
                                       \ alpha stack to the message
                                       \ board.

  Prompt                               \ Execution is stopped
                                       \ temporarily.  Any input is
                                       \ interpreted

  Message('Enter a function name.')    \ Prompt the user again.

  ReadLine Integrate("");              \ The "function name" is now
                                       \ provided to INTEGRATE.

: Sol                                  \ As with INTG, this is an
                                       \ alternate version of SOLVE
                                       \ which will prompt for input.
  Message('Enter an initial guess.')
  prompt
  Message('Enter function name.')
  ReadLine
  Solve("");

Document Intg
Intg ( --- x )

This is the function-key version of integrate.  You do not need to
enter any parameters on a command line.  Instead, you need only press
the function key marked "Intg" and you will be prompted for any
necessary input.

To select a method for integration, first press the function key
marked~Trap to use the trapezoidal rule, or~Simp to use Simpson's
rule.

For further information on the command line version of this
application, see~Integrate $

Document Sol
Sol ( --- x )

This is the function-key version of Solve.  You do not need to enter
any parameters on a command line.  Instead, you need only press the
function key marked "Sol" and you will be prompted for any necessary
input.

There are a couple options for displaying intermediate steps.  Press
the function key marked~Slow to stop execution at each iteration and
wait for a key to be pressed.  Or else, press~Fast to have execution
continue on at each iteration.

For further information on the command line version of this program,
and for further discussion, see~Solve $

: eSimp Explain("Simp");   invisible \ These "e" words will later be
: eTrap Explain("Trap");   invisible \ assigned to the shifted keys.
: eIntg Explain("Intg");   invisible \ Their only function will be to
: eFast Explain("Fast");   invisible \ provide on-line documention for
                                     \ the word assigned to the same
: eSlow Explain("Slow");   invisible \ function key unshifted.
: eSetD Explain("SetDel"); invisible
: eSol  Explain("Sol");    invisible

: Notes Explanation ;        \ This word will be used to display the
                             \ opening comments for the application when
                             \ it is first started up.

                             \ EXPLANATION is similar to EXPLAIN.  It
                             \ will simply explain the word which
                             \ contains it.  When NOTES is executed, it
                             \ will display its own documentation.

: Startup
  1 --> keyset
  Blk("Math")
  1 Assign("Notes")
  2 Assign("Simp")   12 Assign("eSimp")
  3 Assign("Trap")   13 Assign("eTrap")
  4 Assign("Intg")   14 Assign("eIntg")

  5 Assign("Fast")   15 Assign("eFast")
  6 Assign("Slow")   16 Assign("eSlow")
  7 Assign("SetDel") 17 Assign("eSetD")
  8 Assign("Sol")    18 Assign("eSol")

  Notes ;

Document Notes
This package provides two application programs.  For discussion on any
of the following words, either select it here, of press the associated
function key together with the <shift> key to execute eSimp eTrap ...

Integrate a function:

  ~Simp   :  Use Simpson's rule for integration
  ~Trap   :  Use the trapezoidal rule for integration
  ~Intg   :  Start the integration routine and be prompted for input.

Solve for zeroes of a function:

  ~Fast   :  Do not stop after each iteration.
  ~Slow   :  Stop after each iteration and wait for a key to be pressed.
  ~SetDel :  Adjust the value which is added to your initial guess to
             obtain two starting values for the algorithm
  ~Sol    :  Start the Solve routine and be prompted for input. $


\ Now for the final step - to save the compiled application in a
\ package.  Documentation text is written to a file called  PACKAGE.HLP.
\ The following command writes the contents of the dictionary to a file
\ called PACKAGE.PKG.
\
\ It is important that these files keep the same file name extensions
\ (.HLP & .PKG) but you should otherwise rename these files after you
\ exit from Polymath.  For example,
\
\           Rename Package.hlp Solve.hlp
\           Rename Package.pkg Solve.pkg
\
\ The application may then be started up by entering the package name on
\ the DOS command line when Polymath is first started up:
\
\           Pm Solve
\
\ or from within Polymath with:
\
\           Load Solve


  Save
