/* --------------------------------------------------------------------
	Module:		GDBIO.C
	Author:		Craig Payne,
				Compuserve:72672,3706
				Internet:72672.3706@compuserve.com
	Subject:	Imports and exports quoted, comma-delimited data to and
				from HP 100LX database files.
				Base on EXP100.C and GDB.H by Gilles Kohl (Internet:
				gilles@spam.fido.de, FIDO 2:249/6.3, CIS:100114,3146)

	History:	07/27/93	- First version, 0.8

				08/02/93	- Fixed non-growth of data record in WriteRecord
							- Now skips spaces after comma in ImportRecord
							- Absent fields are set to default values in
							  ImportRecord

				08/02/93	- Special HJS release, version, 0.81

				08/09/93	- Fixed zero-length (well, 6) record in ReadRec

				08/09/93	- Special HJS release, version, 0.82

				08/13/93	- Fixed very long lines and fields
							- Fixed first field group box bug
							- generate catagory list (Harry's bug)
							- sets RecNum one "too low"
							- restructured the code for easy addition
							  of other imp/exp filters
							- support un-indexed empty template db
							  (create with copy in database app)
							- released as 0.9

				08/13/93	- fix duplicated category values
							- released as 0.91

				08/18/93	- structure option shows current categories
							- enforced 256 and 128 char limits on categories
							- released as 0.92

				08/23/93	- fix multi-category bug
							- map LICS <> IBM Extended ASCII
							- released as 0.93

				11/01/93	- added "A" option to disable LICS mapping
							- released as 0.94

				12/30/93	- added code in ImportRecord() to ignore extra
							  fields.
							- removed "A" option
							- if'ed-out character mapping logic
							- released as 0.95

	To Do:

	- limit imported fields based on logicalSize in packedWindow in
	  carddef.
	- optional template for output:
	  - format for wp merge, nice reports, etc
	  - seperate program over engine?
	- generate viewptr 0 table
	- set hash=0 only if more that 1 viewpt
	- mention POM in docs
	- GDBLOAD-compatible booleans
	- accept 0x0d, 0x0d 0x0a and 0x0a as line delimiters
	- range-check button settings
	- support CPack YYYYMMDD date format
	- what does CPack do with CrLf in Notes?
	- Mingfang's export format
	- Lotus ASCII import/export format
	- fixed column export format: 11112233345555
	- work with special ?db formats
	- rebuild the index and "type first" table in GDBIO for in and out files
	- support xBase (*.DBF) import/export
	- support user-defined field mapping between the imp/exp and GDB files
	- support importing of notes
	- new, empty GDB files sometimes do not have lookup and first-type tables
	- options on program name don't work
	- @cmd file? for all input data
	- whole issue of deleted fields and others
	- what if Lookup table is > 65 Kbytes, 8192 records?
	- make redirection optional
	- new syntax:

	import: GDBIO GDBi CDFi GDBo {MAP}
	export: GDBIO GDBi {CDFo {MAP} }

	/D{Y|D|M|c} - "DDMMYY", "YY.MM.DD", "YYYYMMDD", etc
	/Q{n|c} use ASCII char n for quote, /Q says no-quote char "{"="{..}", etc
	/C{n|c} use ASCII char n for comma, /C says no-comma char
	/1 - one line per field, n per note
	/MS merge fields with space between - deblank
	/MC merge fields with comma between - deblank
	/MB merge fields with comma and space between - deblank
	/MN merge fields with no deblank or delimiter

	mapping file syntax:

	n{,n}={n|name}

	Done:
	- option to disable LICS mapping

	Testing:
	- test with input db w/lots of deleted records
	- test with big files
	- test with too many and few fields
	- test out-of-disk-space behaviour
	- test with big lines and/or big fields
	- test with empty db as template


	-------------------------------------------------------------------- */

/* --------------------------------------------------------------------
								Includes
	-------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include "gdb.h"


/* --------------------------------------------------------------------
								Defines
	-------------------------------------------------------------------- */

#define DEBUG 0
#define INDEX 0
#define ASCII_OPT 0

#define NoLICS 191
#define NoASCII 168

/* We use this internally to mark the first radio button in a group */
#define FIRST_RADIO FIELDDEF_UNUSED_A

/* The highest offset of radio button data checked for duplicate display */
#define CHECKED_OFFSET_LIMIT 200

// Number of entries we add to the Lookup array in TYPE_INDEX when we
// have to grow it.
#define RECORD_CHUNKS 50

// Maximum length of a category field in a data record
#define MAX_CAT_FIELD 128

// Maximum length of the "all-categories" record
#define MAX_CAT_RECORD 256


/* --------------------------------------------------------------------
								Types
	-------------------------------------------------------------------- */

typedef enum { FALSE, TRUE } BOOLEAN;

typedef union {
	BOOLEAN			boolean;
	char			*pStr;
	unsigned int	word;
	unsigned char	date[3];	// year 0 = 1900, month 0 = jan, day 0 = 1st
} FIELD_DATA;

typedef struct {
	int		MaxRecs;	// Current allocated size of Lookup array in records
	int		TrueSize;	// Actual number of used records in array,
						// always <= MaxRecs
	LOOKUP	*Lookup;	// points to array of entries for this type of record
} TYPE_INDEX;


/* --------------------------------------------------------------------
								Globals
	-------------------------------------------------------------------- */
GDB pDb;     /* pointer to open database descriptor */

BOOLEAN IndexOpt;	// generate index table in created database
BOOLEAN ExportOpt;	// export the records in the input database
#if ASCII_OPT
BOOLEAN ExtAscOpt;  // Use only Extended ASCII, disables LICS mapping,
#endif
BOOLEAN NotesOpt;	// export the notes in the input database
BOOLEAN StructOpt;	// list the db header and fields in the input database
BOOLEAN ImportOpt;	// (internal) add comma-delimited to input GDB to
					// create outGDB

char *FldTypes[] = { /* Field type name array for internal use */
	"bytebool ", "wordbool ", "string   ", "phone    ", "number   ",
	"currency ", "category ", "time     ", "date     ", "radio    ",
	"note     ", "group    ", "static   ", "multiline", "list     ",
	"combo    ", "user     "
};

#if ASCII_OPT
// Lotus International Character Set (LICS) to IBM Extended ASCII
// Based on the table on pages C-5 through C-7 of the 100LX users guide
unsigned char ToASCII[256] = {

	NoASCII,	//  0
	NoASCII,	//  1
	NoASCII,	//  2
	NoASCII,	//  3
	NoASCII,	//  4
	NoASCII,	//  5
	NoASCII,	//  6
	NoASCII,	//  7

	NoASCII,	//  8
	NoASCII,	//  9
	NoASCII,	//  10
	NoASCII,	//  11
	NoASCII,	//  12
	NoASCII,	//  13
	NoASCII,	//  14
	NoASCII,	//  15

	NoASCII,	//  16
	NoASCII,	//  17
	NoASCII,	//  18
	NoASCII,	//  19
	NoASCII,	//  20
	NoASCII,	//  21
	NoASCII,	//  22
	NoASCII,	//  23

	NoASCII,	//  24
	NoASCII,	//  25
	NoASCII,	//  26
	NoASCII,	//  27
	NoASCII,	//  28
	NoASCII,	//  29
	NoASCII,	//  30
	NoASCII,	//  31

	' ',		//  32
	'!',		//  33
	'"',		//  34
	'#',		//  35
	'$',		//  36
	'%',		//  37
	'&',		//  38
	'\'',		//  39

	'(',		//  40
	')',		//  41
	'*',		//  42
	'+',		//  43
	',',		//  44
	'-',		//  45
	'.',		//  46
	'/',		//  47

	'0',		//  48
	'1',		//  49
	'2',		//  50
	'3',		//  51
	'4',		//  52
	'5',		//  53
	'6',		//  54
	'7',		//  55

	'8',		//  56
	'9',		//  57
	':',		//  58
	';',		//  59
	'<',		//  60
	'=',		//  61
	'>',		//  62
	'?',		//  63

	'@',		//  64
	'A',		//  65
	'B',		//  66
	'C',		//  67
	'D',		//  68
	'E',		//  69
	'F',		//  70
	'G',		//  71

	'H',		//  72
	'I',		//  73
	'J',		//  74
	'K',		//  75
	'L',		//  76
	'M',		//  77
	'N',		//  78
	'O',		//  79

	'P',		//  80
	'Q',		//  81
	'R',		//  82
	'S',		//  83
	'T',		//  84
	'U',		//  85
	'V',		//  86
	'W',		//  87

	'X',		//  88
	'Y',		//  89
	'Z',		//  90
	'[',		//  91
	'\\',		//  92
	']',		//  93
	'^',		//  94
	'_',		//  95

	'`',		//  96
	'a',		//  97
	'b',		//  98
	'c',		//  99
	'd',		// 100
	'e',		// 101
	'f',		// 102
	'g',		// 103

	'h',		// 104
	'i',		// 105
	'j',		// 106
	'k',		// 107
	'l',		// 108
	'm',		// 109
	'n',		// 110
	'o',		// 111

	'p',		// 112
	'q',		// 113
	'r',		// 114
	's',		// 115
	't',		// 116
	'u',		// 117
	'v',		// 118
	'w',		// 119

	'x',		// 120
	'y',		// 121
	'z',		// 122
	'{',		// 123
	'|',		// 124
	'}',		// 125
	'~',		// 126
	'',		// 127

	'`',		// 128
	'\'',		// 129
	'^',		// 130
	NoASCII,	// 131 - bare umlaut
	'~',		// 132
	NoASCII,	// 133
	NoASCII,	// 134
	NoASCII,	// 135

	NoASCII,	// 136
	NoASCII,	// 137
	NoASCII,	// 138
	NoASCII,	// 139
	NoASCII,	// 140
	NoASCII,	// 141
	NoASCII,	// 142
	NoASCII,	// 143

	'`',		// 144
	'\'',		// 145
	'^',		// 146
	NoASCII,	// 147 - bare umlaut
	'~',		// 148
	NoASCII,	// 149
	'_',		// 150
	0x1e,		// 151 - up triangle

	0x1f,		// 152 - down triangle
	NoASCII,	// 153
	' ',		// 154
	0x1b,		// 155 - left arrow
	NoASCII,	// 156
	NoASCII,	// 157
	NoASCII,	// 158
	NoASCII,	// 159

	'',		// 160 - Guilder
	'',		// 161
	'',		// 162
	'',		// 163
	'"',		// 164
	'',		// 165
	'',		// 166
	0x15,		// 167 - section sign

	NoASCII,	// 168 - XO
	NoASCII,	// 169 - circle-C, copyright
	NoASCII,	// 170
	'',		// 171
	NoASCII,	// 172
	'',		// 173
	'',		// 174
	'',		// 175

	'',		// 176
	'',		// 177
	'',		// 178
	NoASCII,	// 179
	'"',		// 180
	'',		// 181
	0x14,		// 182 - paragraph
	'',		// 183

	NoASCII,	// 184
	NoASCII,	// 185
	NoASCII,	// 186
	'',		// 187
	'',		// 188
	'',		// 189
	'',		// 190
	'',		// 191

	'A',		// 192
	'A',		// 193
	'A',		// 194
	'A',		// 195
	'',		// 196
	'',		// 197
	'',		// 198
	'',		// 199

	'E',		// 200
	'',		// 201
	'E',		// 202
	'E',		// 203
	'I',		// 204
	'I',		// 205
	'I',		// 206
	'I',		// 207

	'D',		// 208
	'',		// 209
	'O',		// 210
	'O',		// 211
	'O',		// 212
	'O',		// 213
	'',		// 214
	'O',		// 215

	'0',		// 216
	'U',		// 217
	'U',		// 218
	'U',		// 219
	'',		// 220
	'Y',		// 221
	'P',		// 222
	'',		// 223

	'',		// 224
	'a',		// 225
	'',		// 226
	'a',		// 227
	'',		// 228
	'',		// 229
	'',		// 230
	'',		// 231

	'',		// 232
	'',		// 233
	'',		// 234
	'',		// 235
	'',		// 236
	'',		// 237
	'',		// 238
	'',		// 239

	'd',		// 240
	'',		// 241
	'',		// 242
	'',		// 243
	'',		// 244
	'o',		// 245
	'',		// 246
	'o',		// 247

	'o',		// 248
	'',		// 249
	'',		// 250
	'',		// 251
	'',		// 252
	'',		// 253
	'p',		// 254
	''			// 255
};

unsigned char ToLICS[256] = {

	NoLICS,	//   0
	NoLICS,	//   1
	NoLICS,	//   2
	NoLICS,	//   3
	NoLICS,	//   4
	NoLICS,	//   5
	NoLICS,	//   6
	NoLICS,	//   7

	NoLICS,	//   8
	NoLICS,	//   9
	NoLICS,	//  10
	NoLICS,	//  11
	NoLICS,	//  12
	NoLICS,	//  13
	NoLICS,	//  14
	NoLICS,	//  15

	NoLICS,	//  16
	NoLICS,	//  17
	NoLICS,	//  18
	NoLICS,	//  19
	182,	//  20  
	167,	//  21  
	NoLICS,	//  22
	NoLICS,	//  23

	NoLICS,	//  24
	NoLICS,	//  25
	NoLICS,	//  26
	155,	//  27  
	NoLICS,	//  28
	NoLICS,	//  29
	151,	//  30  
	152,	//  31  

	' ',	//  32
	'!',	//  33
	'"',	//  34
	'#',	//  35
	'$',	//  36
	'%',	//  37
	'&',	//  38
	'\'',	//  39

	'(',	//  40
	')',	//  41
	'*',	//  42
	'+',	//  43
	',',	//  44
	'-',	//  45
	'.',	//  46
	'/',	//  47

	'0',	//  48
	'1',	//  49
	'2',	//  50
	'3',	//  51
	'4',	//  52
	'5',	//  53
	'6',	//  54
	'7',	//  55

	'8',	//  56
	'9',	//  57
	':',	//  58
	';',	//  59
	'<',	//  60
	'=',	//  61
	'>',	//  62
	'?',	//  63

	'@',	//  64
	'A',	//  65
	'B',	//  66
	'C',	//  67
	'D',	//  68
	'E',	//  69
	'F',	//  70
	'G',	//  71

	'H',	//  72
	'I',	//  73
	'J',	//  74
	'K',	//  75
	'L',	//  76
	'M',	//  77
	'N',	//  78
	'O',	//  79

	'P',	//  80
	'Q',	//  81
	'R',	//  82
	'S',	//  83
	'T',	//  84
	'U',	//  85
	'V',	//  86
	'W',	//  87

	'X',	//  88
	'Y',	//  89
	'Z',	//  90
	'[',	//  91
	'\\',	//  92
	']',	//  93
	'^',	//  94
	'_',	//  95

	'`',	//  96
	'a',	//  97
	'b',	//  98
	'c',	//  99
	'd',	// 100
	'e',	// 101
	'f',	// 102
	'g',	// 103

	'h',	// 104
	'i',	// 105
	'j',	// 106
	'k',	// 107
	'l',	// 108
	'm',	// 109
	'n',	// 110
	'o',	// 111

	'p',	// 112
	'q',	// 113
	'r',	// 114
	's',	// 115
	't',	// 116
	'u',	// 117
	'v',	// 118
	'w',	// 119

	'x',	// 120
	'y',	// 121
	'z',	// 122
	'{',	// 123
	'|',	// 124
	'}',	// 125
	'~',	// 126
	'',	// 127

	199,	// 128  
	252,	// 129  
	233,	// 130  
	226,	// 131  
	228,	// 132  
	224,	// 133  
	229,	// 134  
	231,	// 135  

	234,	// 136  
	235,	// 137  
	232,	// 138  
	239,	// 139  
	238,	// 140  
	236,	// 141  
	196,	// 142  
	197,	// 143  

	201,	// 144  
	230,	// 145  
	198,	// 146  
	244,	// 147  
	246,	// 148  
	242,	// 149  
	251,	// 150  
	249,	// 151  

	253,	// 152  
	214,	// 153  
	220,	// 154  
	162,	// 155  
	163,	// 156  
	165,	// 157  
	166,	// 158  
	160,	// 159  

	NoLICS,	// 160
	237,	// 161  
	243,	// 162  
	250,	// 163  
	241,	// 164  
	209,	// 165  
	NoLICS,	// 166
	NoLICS,	// 167

	191,	// 168  
	NoLICS,	// 169
	NoLICS,	// 170
	189,	// 171  
	188,	// 172  
	161,	// 173  
	171,	// 174  
	187,	// 175  

	NoLICS,	// 176
	NoLICS,	// 177
	NoLICS,	// 178
	NoLICS,	// 179
	NoLICS,	// 180
	NoLICS,	// 181
	NoLICS,	// 182
	NoLICS,	// 183

	NoLICS,	// 184
	NoLICS,	// 185
	NoLICS,	// 186
	NoLICS,	// 187
	NoLICS,	// 188
	NoLICS,	// 189
	NoLICS,	// 190
	NoLICS,	// 191

	NoLICS,	// 192
	NoLICS,	// 193
	NoLICS,	// 194
	NoLICS,	// 195
	NoLICS,	// 196
	NoLICS,	// 197
	NoLICS,	// 198
	NoLICS,	// 199

	NoLICS,	// 200
	NoLICS,	// 201
	NoLICS,	// 202
	NoLICS,	// 203
	NoLICS,	// 204
	NoLICS,	// 205
	NoLICS,	// 206
	NoLICS,	// 207

	NoLICS,	// 208
	NoLICS,	// 209
	NoLICS,	// 210
	NoLICS,	// 211
	NoLICS,	// 212
	NoLICS,	// 213
	NoLICS,	// 214
	NoLICS,	// 215

	NoLICS,	// 216
	NoLICS,	// 217
	NoLICS,	// 218
	NoLICS,	// 219
	NoLICS,	// 220
	NoLICS,	// 221
	NoLICS,	// 222
	NoLICS,	// 223

	NoLICS,	// 224
	223,	// 225  
	NoLICS,	// 226
	173,	// 227  
	NoLICS,	// 228
	NoLICS,	// 229
	181,	// 230  
	NoLICS,	// 231

	NoLICS,	// 232
	NoLICS,	// 233
	NoLICS,	// 234
	NoLICS,	// 235
	NoLICS,	// 236
	NoLICS,	// 237
	NoLICS,	// 238
	NoLICS,	// 239

	NoLICS,	// 240
	177,	// 241  
	174,	// 242  
	190,	// 243  
	NoLICS,	// 244
	NoLICS,	// 245
	175,	// 246  
	NoLICS,	// 247

	176,	// 248  
	NoLICS,	// 249
	183,	// 250  
	NoLICS,	// 251
	NoLICS,	// 252
	178,	// 253  
	131,	// 254  
	NoLICS	// 255
};
#endif

/* --------------------------------------------------------------------
								Utility functions
	-------------------------------------------------------------------- */

/****************************************************************************

 StrLower: Converts a string to lower case in-place

****************************************************************************/

char *StrLower(char *pString) {

	while (*pString = (char)tolower(*pString)) {pString++;}

	return pString;

} // strLower

#if DEBUG
void DumpBytes(
	unsigned char	*pBytes,
	unsigned long	Length,
	unsigned long	StartingOffset
) {

#define BytesPerLine 16

	unsigned		i;
	unsigned long	LineStart;
	unsigned		BytesInLine;
	char			c;

	LineStart = 0;
	while (LineStart < Length) {

		BytesInLine = min(BytesPerLine, (unsigned) (Length - LineStart));

		printf("%08lx:", StartingOffset + LineStart);

		// dump a line in hex
		for (i = 0; i < BytesInLine; i++) {
			printf(" %02x", *(pBytes+LineStart+i));
		}

		// pad with spaces between hex and ASCII dumps
		printf("%*s  [", (BytesPerLine - BytesInLine)*3, "");

		// dump a line interpreted as ASCII
		for (i = 0; i < BytesInLine; i++) {

			c = *(pBytes+LineStart+i);

			printf("%c", ( (c < 32) || (c > 127) ) ? '.': c);

		}

		printf("]\n");

		LineStart += BytesInLine;

	} // while

	printf("\n");

} // DumpBytes

void ShowRecHdr(RECORDHEADER *pHdr) {

   printf("Type: (%2d) ", pHdr->Type);
   switch (pHdr->Type) {
      case TYPE_DBHEADER:	 printf("DbHeader   "); break;
      case TYPE_CARDDEF:	 printf("CardDef    "); break;
      case TYPE_CATEGORY:	 printf("Category   "); break;
      case TYPE_FIELDDEF:	 printf("FieldDef   "); break;
      case TYPE_VIEWPTDEF:	 printf("ViewptDef  "); break;
      case TYPE_NOTE:		 printf("Note       "); break;
      case TYPE_VIEWPTTABLE: printf("ViewptTable"); break;
      case TYPE_DATA:		 printf("Data       "); break;
      case TYPE_LINKDEF:	 printf("LinkDef    "); break;
      case TYPE_CARDPAGEDEF: printf("CardPageDef"); break;
      case TYPE_USER:		 printf("User       "); break;
      case TYPE_LOOKUPTABLE: printf("LookupTable"); break;
	  default:               printf("UNKNOWN    "); break;
   }

   printf(", Stat: (%1d) ", pHdr->Status);
   switch (pHdr->Status) {
      case NORMAL:   printf("Normal  "); break;
	  case GARBAGE:  printf("Garbage "); break;
	  case MODIFIED: printf("Modified"); break;
	  default:       printf("UNKNOWN "); break;
   }

   printf(", Len %4d, Num %4d\n", pHdr->Length, pHdr->Number);

}

#endif

void Fatal(char *Fmt, ...)
/*
 * Output a fatal error message (printf functionality), then bail out.
 * char *Fmt: Format string as in printf()
 *      ... : arguments as in printf()
 */
{
	va_list ArgPtr;

	va_start(ArgPtr, Fmt);
	fprintf(stderr, "Fatal error: ");
	vfprintf(stderr, Fmt, ArgPtr);
	fprintf(stderr, "\n");
	exit(-1);
}

void AddToIndex(
	RECORDHEADER	*pHdr,
	unsigned long	Offset,
	TYPE_INDEX		*TypeIndexArray
) {

	TYPE_INDEX		*pTypeIndex;
	LOOKUP			*pLookupEntry;
	int				NewMaxRecs;

	if (pHdr->Type > TYPE_LOOKUPTABLE)
		Fatal("illegal record type");

	pTypeIndex = &TypeIndexArray[pHdr->Type];

	// Grow the entry ?
	if (pHdr->Number >= pTypeIndex->MaxRecs) {
		NewMaxRecs = pHdr->Number + RECORD_CHUNKS;
		// malloc or realloc ?
		if (pTypeIndex->Lookup) {
			if(!(
				pTypeIndex->Lookup=
					realloc(pTypeIndex->Lookup, sizeof(LOOKUP)*NewMaxRecs)
			))
				Fatal("not enough memory to add to index");
		} else {
			if(!(
				pTypeIndex->Lookup=malloc(sizeof(LOOKUP)*NewMaxRecs)
			))
				Fatal("not enough memory to start index");
		}

		memset(
			&pTypeIndex->Lookup[pTypeIndex->MaxRecs],
			0,
			sizeof(LOOKUP)*RECORD_CHUNKS
		);
		
		pTypeIndex->MaxRecs = NewMaxRecs;

	} // if grow

	pLookupEntry = &pTypeIndex->Lookup[pHdr->Number];

	pTypeIndex->TrueSize = max(pTypeIndex->TrueSize, pHdr->Number+1);
	pLookupEntry->Size = pHdr->Length;
	pLookupEntry->Filters = 0xffff; // dirty for all viewpoints (?)
	pLookupEntry->Flags =
		(char) (pHdr->Status == GARBAGE ? LUFLAG_DELETED : 0);

	// This depends on Intel byte ordering
	memcpy(&pLookupEntry->Offset, &Offset, sizeof(pLookupEntry->Offset));

#if DEBUG
	printf("off %4lx, ", Offset);
	ShowRecHdr(pHdr);
#endif

}

void BuildIndex(
	LOOKUPTABLE **ppLuTab,		// lookup table (includes RecHdr)
	TYPEFIRST   Tf,				// the typefirst table
	TYPE_INDEX	*TypeIndexArray
) {
	int		i;
	int		LookupEntries;
	size_t	LookupSize;
	int		TypeFirst;

	// Calculate entries in complete Lookup table

	// start with one for the lookup table itself plus the RECORDHEADER
	LookupEntries = 1;
	for (i = 0; i < TYPE_LOOKUPTABLE; i++)
		LookupEntries += TypeIndexArray[i].TrueSize;

	LookupSize = sizeof(RECORDHEADER) + LookupEntries*sizeof(LOOKUP);

	if (!(*ppLuTab = malloc(LookupSize)))
		Fatal("not enough memory to build index");

#if DEBUG
	memset(*ppLuTab ,0 , LookupSize);
#endif

	TypeFirst = 0;
	// copy lookup for each type to build master lookup table
	for (i = 0; i < TYPE_LOOKUPTABLE; i++) {
		Tf[i] = TypeFirst;
		if (TypeIndexArray[i].TrueSize) {

#if DEBUG
			printf("Type %d, TypeFirst %d\n", i, TypeFirst);
			DumpBytes(
				(unsigned char *)TypeIndexArray[i].Lookup,
				TypeIndexArray[i].TrueSize*sizeof(LOOKUP),
				0
			);
#endif
			memcpy(
				&(*ppLuTab)->Lookup[TypeFirst],
				TypeIndexArray[i].Lookup,
				TypeIndexArray[i].TrueSize*sizeof(LOOKUP)
			);

#if DEBUG
			DumpBytes(
				(unsigned char *)&(*ppLuTab)->Lookup[TypeFirst],
				8,
				0
			);
#endif
			TypeFirst += TypeIndexArray[i].TrueSize;

			if (TypeIndexArray[i].Lookup);
				free(TypeIndexArray[i].Lookup);
			TypeIndexArray[i].MaxRecs = 0;
			TypeIndexArray[i].TrueSize = 0;
			TypeIndexArray[i].Lookup = NULL;
		}
	}

	// the lookup table record is the last one
	Tf[TYPE_LOOKUPTABLE] = LookupEntries-1;

	(*ppLuTab)->RecHdr.Type = TYPE_LOOKUPTABLE;
	(*ppLuTab)->RecHdr.Status = MODIFIED;
	(*ppLuTab)->RecHdr.Length = LookupSize;
	(*ppLuTab)->RecHdr.Number = 0; // The one and only Lookup Table record

#if DEBUG
	{

   		int i;

   		DumpBytes(
			(unsigned char *)(*ppLuTab)->Lookup,
			LookupEntries*sizeof(LOOKUP),
			0
		);

    	printf("     First\nType record Count\n");
    	for (i=0; i<32; i++) {
	    	printf("%4d %4d", i, Tf[i]);
			if (i < 31)
	    		printf(" %4d", Tf[i+1]-Tf[i]);
			else
	    		printf("    1");
	    	printf("\n");
		}

	}
#endif

}

void CheckSignature(GDB *pDb)
/* 
 * Read signature and check against what it should be.
 * GDB *pDb: the GDB descriptor pointer. 
 *
 */ 
{
	SIGNATURE Sig;

	/* read signature */
	if(fread(&Sig, sizeof(Sig), 1, pDb->f) != 1) {
		Fatal("Read signature failed");
	}

	/* check against HP-defined signature */
	if(strcmp((char*)&Sig, HPSIGNATURE)) Fatal("Signature mismatch");
}

BOOLEAN ReadRecHdr(GDB *pDb, RECORDHEADER *pHdr)
/*
 * Read record header, exits on error
 * GDB *pDB: the GDB descriptor pointer
 * RECORDHDR *pHdr: where to store header
 */
{
	if (fread(pHdr, sizeof(*pHdr), 1, pDb->f) != 1) {
		if (feof(pDb->f))
		  return FALSE;
	  else
			Fatal("Read record header failed.");
	}
	return TRUE;
}

void *ReadRec(GDB *pDb, void *pDest, unsigned Size)
/*
 * Read a record. 
 *    void *pDest: where to store record. If NULL, storage will be
 *                 malloc'ed. Caller must free in this case
 *  unsigned Size: record size to be expected. If 0, size in record
 *                 header will be used and returned in *pSize.
 * returns void *: pointer to record read in.
 */
{
	int Offset = 0;
	RECORDHEADER Hdr;

	if(!Size) { /* no size given, try to find out */
		if (!ReadRecHdr(pDb, &Hdr))  /* get record header */
		  return NULL;
		Offset = sizeof(Hdr);
		Size = Hdr.Length;
	}
	if(!pDest) { /* no pointer given, malloc requested */
		if(!(pDest = malloc(Size))) Fatal("Out of memory");
	}
	/* if header already read, copy it to destination */
	if(Offset) memcpy(pDest, &Hdr, Offset);
	if (Size-Offset) {
		if(fread((char *)pDest+Offset, Size-Offset, 1, pDb->f) != 1) {
			Fatal("Read record failed");
		}
	}
	return pDest;
}

void Expect(void *pRec, RECORD_TYPE Type)
/*
 * Expect a certain record type. Fatal error on mismatch
 *       void *pRec: points to record or record header.
 * RECORD_TYPE Type: type to be expected
 *
 */
{
	RECORDHEADER *pHdr = pRec; /* more readable than a cast */

	if( (RECORD_TYPE)pHdr->Type != Type) {
		Fatal("Expected record type %d, got %d", Type, pHdr->Type);
	}
}

void GetDbHeader(GDB *pDb)
/*
 * Read DBHEADER structure, check if correct. Store in DBHEADER in 
 * the DB descriptor.
 * GDB *pDb: pointer to DB desciptor
 *
 */
{
	ReadRec(pDb, &pDb->Hdr, sizeof(pDb->Hdr));
	Expect(&pDb->Hdr.RecHdr, TYPE_DBHEADER);
}

void Seek2Record(GDB *pDb, RECORD_TYPE RecType, int Nbr, int *pSize)
/*
 * Position to a certain record in file, given type of record and 
 * record number.
 *            GDB *pDb: pointer to DB desciptor
 * RECORD_TYPE RecType: type of record to position to
 *             int Nbr: number of record
 *          int *pSize: if not NULL, size of record will be stored here.
 *
 */
{
	long Offset = 0L;
	unsigned RecNo;

	RecNo = pDb->Tf[RecType]+(unsigned)Nbr;
	/* tricky programming here. INTEL-dependent */
	memcpy(&Offset, &(pDb->pIdx[RecNo].Offset), 3);
	if(pSize) *pSize = pDb->pIdx[RecNo].Size;
	fseek(pDb->f, Offset, SEEK_SET);
}

void *GetRecord(GDB *pDb, void *pDest, RECORD_TYPE RecType, int Nbr)
/* 
 *  Position to and read a record given type and number.
 *
 *            GDB *pDb: pointer to DB desciptor
 *         void *pDest: where to store record. If NULL, storage will be
 *                      malloc'ed. Caller must free in this case
 * RECORD_TYPE RecType: type of record to position to
 *             int Nbr: number of record
 *      returns void *: pointer to record read in. NULL if record 0 sized.
 */
{
	int Size;
	void *p;

	Seek2Record(pDb, RecType, Nbr, &Size);
	if(!Size) return NULL;
	p = ReadRec(pDb, pDest, Size);
	Expect(p, RecType);
	return p;
}

void GenIndex(GDB *pDb) {

	TYPE_INDEX		TypeIndexArray[32] = {0};
	RECORDHEADER	*pRec;
	unsigned long	RecOffset;

#if DEBUG
	fprintf(stderr, "Warning: index not found in GDB input file.\n");
#endif

	if (fseek(pDb->f, sizeof(SIGNATURE), SEEK_SET))
		Fatal("error seeking past signature");

	while (!feof(pDb->f)) {
		pRec = NULL;
		RecOffset = ftell(pDb->f);
		pRec = ReadRec(pDb, pRec, 0);
		if (pRec) {

			AddToIndex(pRec, RecOffset, TypeIndexArray);
			free(pRec);
		}
	}

	BuildIndex(&pDb->pLuTab, pDb->Tf, TypeIndexArray);

	// point into LOOKUPTABLE record
	pDb->pIdx = pDb->pLuTab->Lookup;
}

void GetIndex(GDB *pDb)
/*
 *  Read database index (Lookuptable)
 *  NOTE: The lookuptable _must_ be present.
 *  GDB *pDb: pointer to DB desciptor
 *
 */
{
	/* seek to index (LookupTable) */
	if(!pDb->Hdr.LookupSeek) {
		// Fatal("Missing loopkup table. Db not closed ?");
		GenIndex(pDb);
	} else {
		if(fseek(pDb->f, pDb->Hdr.LookupSeek, SEEK_SET)) {
			Fatal("Seek to lookup failed");
		}

		pDb->pLuTab = ReadRec(pDb, NULL, 0);
		Expect(pDb->pLuTab, TYPE_LOOKUPTABLE);

		/* point into LOOKUPTABLE record */
		pDb->pIdx = pDb->pLuTab->Lookup;

		/* get TypeFirst table */
		if(fread(pDb->Tf, sizeof(pDb->Tf), 1, pDb->f) != 1) {
			Fatal("Read typefirst failed");
		}
	}
}

void ShowFieldDefs(GDB *pDb)
/*
 * Show information on a field descriptor.
 * FIELDDEF *pFld: pointer to field definition structure
 *
 */
{
	int LastType=-1;
	int DataFldNum=1;
	int FldNum;
	FIELDDEF *pFld;

	printf("Fld Name                 Type      Off Reserved     Flags\n");

	for (pFld=pDb->pFld, FldNum=0; FldNum < pDb->FldCnt; pFld++, FldNum++) {
		if (
			LastType == RADIO_FIELD &&
			pFld->FieldType != RADIO_FIELD &&
			pFld->FieldType != GROUP_FIELD
		) {
			DataFldNum++;
			printf("\n");
		}
		if (pFld->Flags & FIELDDEF_NODATA) {
			if (pFld->FieldType == GROUP_FIELD)
				printf("\n    %-20s group\n", pFld->Name);
		} else {
			printf(
				"%3d %-20s %s %3d",
				DataFldNum,
				pFld->Name, 
				pFld->FieldType < 17 ? FldTypes[pFld->FieldType] : "?unknown?",
				pFld->DataOffset
			);
			switch (pFld->FieldType) {
				case BYTEBOOL_FIELD:
				case WORDBOOL_FIELD:
					printf(" Mask=0x%04x ", pFld->Reserved);
					break;
				case RADIO_FIELD:
					printf(" Value=%-5d", pFld->Reserved);
					break;
				case CATEGORY_FIELD:
					printf(" CatRec=%-5d", pFld->Reserved);
					break;
				default:
					printf("             ");
			}
			if (pFld->Flags & FIELDDEF_NODATA)		printf(" NoData");
			if (pFld->Flags & FIELDDEF_RESERVED)	printf(" Reserved"); 
			if (pFld->Flags & FIELDDEF_RELATIVE)	printf(" Relative");
			if (pFld->Flags & FIELDDEF_NULLTITLE)	printf(" NullTitle");
			printf("\n");
			if (pFld->FieldType != RADIO_FIELD)	DataFldNum++;
		}
		LastType=pFld->FieldType;
	}
}

void GetFields(GDB *pDb)
/*
 * load the array of field definitions and store in DB descriptor
 * GDB *pDb: pointer to DB desciptor
 * 
 */
{
/* The highest offset of radio button data checked for grouping buttons */
#define CHECKED_OFFSET_LIMIT 200

	/* used to mark just one radio button per group */
	BOOLEAN OffsetUsed[CHECKED_OFFSET_LIMIT];
	FIELDDEF *pFld;

	int i;

	/* Clear flag that indicates offset has been used */
	memset(OffsetUsed, 0, sizeof(OffsetUsed));

	// compute number of fields
	for (
		i = pDb->Tf[TYPE_FIELDDEF], pDb->FldCnt = 0;
		i < pDb->Tf[TYPE_FIELDDEF+1];
		i++
	)
		if (!(pDb->pIdx[i].Flags & LUFLAG_DELETED)) pDb->FldCnt++;
	
	if(!pDb->FldCnt) Fatal("No field definitions found");

	/* allocate memory for field definition array */
	if(!(pDb->pFld = calloc(pDb->FldCnt, sizeof(FIELDDEF)))) {
		Fatal("Out of memory for fielddef table");
	}

	/* Allocate memory for data field pointer array
	   There are always less or equal data fields than fields */

	if(!(pDb->pDataFlds = calloc(pDb->FldCnt, sizeof(FIELDDEF *)))) {
		Fatal("Out of memory for data field table");
	}

	pDb->DataFldCnt = 0;	// count the data fields

	// Initialize total size (in bytes) of fixed (non-relative) fields
	// One because we always end the fixed data with a null. Relative
	// offsets to null strings all point to this one null.
	pDb->FixedSize=1;

	/* load field def table */
	for(
		i = 0, pFld = pDb->pFld;
		i < pDb->FldCnt;
		i++, pFld++
	) {
		
		Seek2Record(pDb, TYPE_FIELDDEF, i, NULL);
		ReadRec(pDb, pFld, sizeof(*pDb->pFld));
		Expect(&pFld->RecHdr, TYPE_FIELDDEF);

		// skip garbage or deleted field records
		if (
			pFld->RecHdr.Status == GARBAGE ||
			pDb->pIdx[pDb->Tf[TYPE_FIELDDEF]+pFld->RecHdr.Number].Flags &
				LUFLAG_DELETED
		) continue;

		// ensure that FIRST_RADIO flag is off
		pFld->Flags &= ~FIRST_RADIO;

		// process only fields with data
		if (!(pFld->Flags & FIELDDEF_NODATA)) {
			/*
			Radio button fields are an oddity. There are multiple fields but
			only one data location. For import and export we only want to
			process one value per group of buttons. To make this easier we
			mark the first button of each group.
			*/

			if (pFld->DataOffset >= CHECKED_OFFSET_LIMIT)
				Fatal("too many fixed fields");

			if (
				(pFld->FieldType == RADIO_FIELD) &&
				!OffsetUsed[pFld->DataOffset]
			) {
				pFld->Flags |= FIRST_RADIO;
				OffsetUsed[pFld->DataOffset] = TRUE;
			}

			// Fill the data field array only with valid data fields
			// including only the first radio button of a group
			if (
				(pFld->Flags & FIRST_RADIO) ||
				(pFld->FieldType != RADIO_FIELD)
			) {
				pDb->pDataFlds[pDb->DataFldCnt++] = pFld;

				// total byte-size of fixed fields and set data filed pointer
				if (pFld->Flags & FIELDDEF_RELATIVE) {
					// all relative fields have a fixed 2-byte offset
					pDb->FixedSize += 2;
				} else {
					switch(pFld->FieldType) {
	 					case DATE_FIELD:
							pDb->FixedSize += 3;
							break;
						case BYTEBOOL_FIELD:
	 						pDb->FixedSize += 1;
							break;
						case RADIO_FIELD:
						case TIME_FIELD:
						case WORDBOOL_FIELD:
						case NOTE_FIELD:
	 						pDb->FixedSize += 2;
							break;
						case STRING_FIELD:
						case COMBO_FIELD:
						case PHONE_FIELD:
						case NUMBER_FIELD:
						case CURRENCY_FIELD:
						case CATEGORY_FIELD:
						case MULTILINE_FIELD:
						case USER_FIELD:
							Fatal("fixed string field");
					}
				} // else fixed field
			} // if valid data field
		} // if data field
	}
#if DEBUG
	{
		FIELDDEF *pFld;
		int n;

		printf("%d data fields\n", pDb->DataFldCnt);

		for(
			n = 0, pFld = pDb->pDataFlds[0];
			n < pDb->DataFldCnt;
			n++, pFld = pDb->pDataFlds[n]
		) { /* loop over data fields */
			printf(
				"%3d %-20s %s %3d",
				n,
				pFld->Name, 
				pFld->FieldType < 17 ? FldTypes[pFld->FieldType] : "?unknown?",
				pFld->DataOffset
			);
			switch (pFld->FieldType) {
				case BYTEBOOL_FIELD:
				case WORDBOOL_FIELD:
					printf(" Mask=0x%04x ", pFld->Reserved);
					break;
				case RADIO_FIELD:
					printf(" Value=%-5d", pFld->Reserved);
					break;
				case CATEGORY_FIELD:
					printf(" CatRec=%-5d", pFld->Reserved);
					break;
				default:
					printf("             ");
			}
			if (pFld->Flags & FIELDDEF_NODATA)		printf(" NoData");
			if (pFld->Flags & FIELDDEF_RESERVED)	printf(" Reserved"); 
			if (pFld->Flags & FIELDDEF_RELATIVE)	printf(" Relative");
			if (pFld->Flags & FIELDDEF_NULLTITLE)	printf(" NullTitle");
			printf("\n");
		}
	}

	printf("Size of fixed fields is %d\n", pDb->FixedSize);
#endif
	if (StructOpt) ShowFieldDefs(pDb);
}

void ShowOneField(
	unsigned char *pBytes,
	FIELDDEF *pFld,
	int *pNoteNum
) {
#if ASCII_OPT
	unsigned char *p;
#endif
	unsigned *pWords = (unsigned *)pBytes;

	switch(pFld->FieldType) {
		case STRING_FIELD:
		case COMBO_FIELD:
		case PHONE_FIELD:
		case NUMBER_FIELD:
		case CURRENCY_FIELD:
		case CATEGORY_FIELD:
	 		// all the above fields are stored as strings.

#if ASCII_OPT
			if (!ExtAscOpt) {
				// Convert string from LICS to ASCII
				for (p = pBytes; *p; p++)
					*p = ToASCII[(unsigned char)*p];
			}
#endif

	 		printf("\"%s\"", pBytes);
	 		break;
		case TIME_FIELD:
			if (*pWords >= 24*60)
	 			printf("\"\"");	// blank or invalid time
			else
	 			printf("\"%02d:%02d\"", *pWords/60, *pWords%60);
	 		break;
		case DATE_FIELD:
	 		/* Note: mm/dd/yy date format used here */
			if ( pBytes[0] == 0xff || pBytes[1] > 11 || pBytes[2] > 30)
				printf("\"\"");	// blank or invalid time
			else
				printf("\"%02d/%02d/%02d\"",
					pBytes[1]+1, pBytes[2]+1, pBytes[0]);
	 		break;
		case BYTEBOOL_FIELD:	/* Checkbox: points to a bit in a byte */
		case WORDBOOL_FIELD:	/* Checkbox: points to a bit in a word */
			printf("\"%c\"", (*pWords & pFld->Reserved)? 'T' : 'F');
			break;
		case RADIO_FIELD:
			printf("\"%d\"", *pWords);
			break;
		case NOTE_FIELD:		/* The field is a note (which resides in a
								separate record); dataoffset points to the
								record number of the note.  If no note is
								attached, the record number pointed to
								should be -1. */
	 		if (*pNoteNum == -1)
				*pNoteNum = *pWords;
	 		else
				Fatal("record contains more that one note");

#if 0
	 		if ( *pNoteNum == -1 || !NotesOpt)
				printf("\"\"");
			else
				printf("\"%d\"", *pNoteNum);
#else
				printf("\"\"");	// no reason to export note numbers for now
#endif
	 		break;
		case MULTILINE_FIELD:	/* Same as a string, but can have CR/LFs */
	 		printf("<CRLF>");
			break;
		case USER_FIELD:		/* The applications can define their own
								field types starting at USER_FIELD NOTE:
								The FIELDDEF for user defined fields
								MUST have the status bit set to
								FIELDDEF_CALLBACK and a FieldCallback
								function provided. */
	 		printf("<USER>");
			break;
		default:
	 		printf("<UNKNOWN %d>", pFld->FieldType);
			break;
	}
}

void ShowDataFields(GDB *pDb, DATABYTERECORD *pDat)
/* 
 * Show data contents of a record (all fields)
 *              GDB *pDb: pointer to database descriptor
 * DATABYTERECORD *pDat: pointer to record data
 *                 int n: number of field to show
 */
{

	int NoteNum=-1; /* Catches note numbers in a record */

	FIELDDEF *pFld;
	unsigned TrueOffset;
	unsigned char *pBytes;
	int n;
	long CurPos;
	NOTE *pNote;

	if(NotesOpt) printf("D ");
	for(
		n = 0, pFld = pDb->pDataFlds[0];
		n < pDb->DataFldCnt;
		n++, pFld = pDb->pDataFlds[n]
	) { /* loop over data fields */


		if (pFld->Flags & FIELDDEF_RELATIVE)
			TrueOffset = *(unsigned *)(pDat->Fixed + pFld->DataOffset);
		else
			TrueOffset = pFld->DataOffset;

		pBytes = &pDat->Fixed[TrueOffset];

	 	if (n) putchar(',');
		ShowOneField(pBytes, pFld, &NoteNum);

	} // for
	printf("\n");
	 if ( NotesOpt && (NoteNum != -1)) {
		CurPos = ftell(pDb->f);
		Seek2Record(pDb, TYPE_NOTE, NoteNum, NULL);
		pNote = ReadRec(pDb, NULL, 0);
		printf("N ");
		for (n=0; (unsigned)n<pNote->RecHdr.Length-sizeof(RECORDHEADER);n++)
			if (pNote->Note[n] == '\n') {
				putchar(pNote->Note[n]);
				printf("N ");
			} else if (pNote->Note[n] != '\r')
				putchar(pNote->Note[n]);
		printf("\n");
		free(pNote);
		fseek(pDb->f, CurPos, SEEK_SET);
	}
}

void OutputData(GDB *pDb)
/*
 * Loop over all data records, show all fields (that can be output)
 * GDB *pDb: pointer to database descriptor
 *
 */
{
	int i;
	int Nbr;
	DATABYTERECORD *pDat;

	/* compute number of data records */
	Nbr = pDb->Tf[TYPE_DATA+1] - pDb->Tf[TYPE_DATA];

	/* loop over data records */
	for(i = 0; i < Nbr; i++) {
		if(pDat = GetRecord(pDb, NULL, TYPE_DATA, i)) {
			ShowDataFields(pDb, pDat);
			free(pDat);
		}
	}
}

void DefaultData(
	FIELDDEF	*pFld,
	FIELD_DATA	*pFieldData
) {

	switch(pFld->FieldType) {
		case DATE_FIELD:
			pFieldData->date[0] = 0xff;
			pFieldData->date[1] = 0xff;
			pFieldData->date[2] = 0xff;
			break;

		case BYTEBOOL_FIELD:
		case WORDBOOL_FIELD:
			pFieldData->boolean = FALSE;
			break;

		case TIME_FIELD:
			pFieldData->word = 0x8000;
			break;

		case RADIO_FIELD:
			pFieldData->word = 0;
			break;

		case NOTE_FIELD:
			pFieldData->word = 0xffff;	// undefined notes are marked with 0xffff
			break;

		case STRING_FIELD:
		case COMBO_FIELD:
		case PHONE_FIELD:
		case NUMBER_FIELD:
		case CURRENCY_FIELD:
		case CATEGORY_FIELD:
		case MULTILINE_FIELD:
		case USER_FIELD:
			pFieldData->pStr = 0;
			break;
	} // switch

} // DefaultData

void EncodeData(
	char			*Token,
	FIELDDEF		*pFld,
	FIELD_DATA		*pFieldData,
	unsigned int	InpRecNum
) {

#if ASCII_OPT
	unsigned char *pSrc;
	unsigned char *pDst;
#endif

	char HourStr[3];
	char MinStr[3];
	char Date1[3];
	char Date2[3];
	char Date3[3];
	int Hour;
	int Min;
	int dd;
	int mm;
	int yy;

	HourStr[2]=0;
	MinStr[2]=0;
	Date1[2]=0;
	Date2[2]=0;
	Date3[2]=0;

	if ( Token && *Token ) {

		switch(pFld->FieldType) {
			case DATE_FIELD:
				if (!(
					strlen(Token) == 8 &&
					isdigit(Date1[0] = Token[0]) &&
					isdigit(Date1[1] = Token[1]) &&
					(Token[2] == '/' || Token[2] == '.') &&
					isdigit(Date2[0] = Token[3]) &&
					isdigit(Date2[1] = Token[4]) &&
					(Token[5] == '/' || Token[5] == '.') &&
					isdigit(Date3[0] = Token[6]) &&
					isdigit(Date3[1] = Token[7]) &&
					(Token[2] == Token[5])
				))
					Fatal("invalid date %s in record %d", Token, InpRecNum);

				if (Token[2] == '/') {
					// mm/dd/yy format
					mm = atoi(Date1)-1;
					dd = atoi(Date2)-1;
					yy = atoi(Date3);
					if ( mm > 11 || dd > 30 )
						Fatal("invalid date %02d/%02d/%02d in record %d",
							mm, dd, yy, InpRecNum);
				} else {
					// dd.mm.yy format
					dd = atoi(Date1)-1;
					mm = atoi(Date2)-1;
					yy = atoi(Date3);
					if ( mm > 11 || dd > 30 )
						Fatal("invalid date %02d.%02d.%02d in record %d",
							dd, mm, yy, InpRecNum);
				}

				pFieldData->date[0] = (unsigned char)yy;
				pFieldData->date[1] = (unsigned char)mm;
				pFieldData->date[2] = (unsigned char)dd;
				break;

			case BYTEBOOL_FIELD:
			case WORDBOOL_FIELD:
				pFieldData->boolean = (unsigned char) (
					(Token[0] == 't') ||
					(Token[0] == 'T') ||
					(Token[0] == 'y') ||
					(Token[0] == 'Y')
				);

				break;

			case TIME_FIELD:
				if (!(
					strlen(Token) == 5 &&
					isdigit(HourStr[0] = Token[0]) &&
					isdigit(HourStr[1] = Token[1]) &&
					Token[2] == ':' &&
					isdigit(MinStr[0] = Token[3]) &&
					isdigit(MinStr[1] = Token[4])
				))
					Fatal("invalid time %s in record %d", Token, InpRecNum);

				Hour = atoi(HourStr);
				Min = atoi(MinStr);

				if ( Hour > 23 || Min > 59)
					Fatal("invalid time %02d:%02d in record %d",
						Hour, Min, InpRecNum);

				pFieldData->word = Hour*60+Min;
				break;

			case RADIO_FIELD:
				pFieldData->word = atoi(Token);
				break;

			case NOTE_FIELD:
#if 0
				pFieldData->word = atoi(Token);
#else
				// undefined notes are marked with 0xffff
				pFieldData->word = 0xffff;
#endif
				break;

			case STRING_FIELD:
			case COMBO_FIELD:
			case PHONE_FIELD:
			case NUMBER_FIELD:
			case CURRENCY_FIELD:
			case CATEGORY_FIELD:
			case MULTILINE_FIELD:
			case USER_FIELD:
				if(!(pFieldData->pStr = malloc(strlen(Token)+1)))
					Fatal("Out of memory for string field");


#if ASCII_OPT
				if (ExtAscOpt) {
					strcpy(pFieldData->pStr, Token);
				} else {
					// copy and convert string from ASCII to LICS
					for (
						pSrc = Token, pDst = pFieldData->pStr;
						*pSrc;
						pSrc++, pDst++
					)
						*pDst = ToLICS[(unsigned char)*pSrc];

					*pDst = 0;
				}
#else
				strcpy(pFieldData->pStr, Token);
#endif

//				memcpy(pFieldData->pStr, Token, strlen(Token)+1);

				break;
		} // switch
	} else {
		DefaultData(pFld, pFieldData);
	}
}

BOOLEAN ImportRecord(
	GDB *pDb,
	FILE *InFile,
	unsigned InpRecNum,
	FIELD_DATA	*pDataArray
) {

// The increment that we increase the line buffer size by
#define LINE_INCR 20

	char Tmp[LINE_INCR];
	static char *pLine;
	static unsigned int LineSize=0;
	char *Token;
	char *p;
	char *pEnd;
	int l;
	int InpFldNum;
	int FldNum;
	BOOLEAN *pFieldSet;

	if (!LineSize) {
		LineSize=200; // start with enough room for resonable lines
		if (!(pLine = malloc(LineSize)))
			Fatal("not enough memory for initial line buffer");
	}

	*pLine=0;

	do {
		if( fgets( Tmp, LINE_INCR, InFile ) == NULL )
			if( feof( InFile ) )
				return FALSE;
			else
				Fatal("Error reading import file");
		if (strlen(pLine) + strlen(Tmp) >= LineSize) {
			LineSize += LINE_INCR;
			if (!(pLine = realloc(pLine, LineSize)))
				Fatal("not enough memory to grow line buffer");
		}
		strcat(pLine, Tmp);
	} while (pLine[(l = strlen(pLine))-1] != '\n');

	if(!(pFieldSet = calloc(pDb->DataFldCnt, sizeof(BOOLEAN)))) {
		Fatal("Out of memory for field-set table");
	}

	// zap any trailing newlines or blanks
	p = pLine + l - 1;
	while ( *p == ' ' || *p == '\n') {
		*p = 0;
		// exit if line becomes empty
		if (--p == pLine) return FALSE;
	}

	p = pLine;
	InpFldNum=0;

	while (*p) {

		// if it starts with a quote skip it and look for closing quote
		if (*p == '"') {
			p++;
			// if the closing quote is found change it to a null
			if (pEnd = strchr(p, '"')) {
				*pEnd = 0;
				Token = p;
				p = pEnd+1;
				// if we are not at the end of the line look for next comma
				if (*p) {
					// move past seperating comma
					if (!(p = strchr(p, ',')))
						Fatal("missing comma between fields in line %d", InpRecNum);
					p++;
					// Now skip any spaces after comma and before data
					while (*p == ' ') p++;
				}
			} else {
				Fatal("missing closing quote in line %d", InpRecNum);
			}
		} else { // otherwise look for a comma.
			// if the closing delimiter is found change it to a null
			if (pEnd = strchr(p, ',')) {
				*pEnd = 0;
				Token = p;
				p = pEnd+1;
			} else {
				Token = p;
				p = p + strlen(p);	// point p at final null
			}
		}

		if (InpFldNum < pDb->DataFldCnt) {

			FldNum = InpFldNum;	// one-to-one mapping for now

			EncodeData(
				Token, pDb->pDataFlds[FldNum], &pDataArray[FldNum], InpRecNum
			);

			pFieldSet[FldNum] = TRUE;

#if DEBUG
			printf(
				"InpFldNum %2d, Type %2d, Token [%s]\n",
				InpFldNum, pDb->pDataFlds[FldNum]->FieldType, Token
			);

		} else {
			printf(
				"InpFldNum %2d,          Token [%s] <<< Extra data ignored\n",
				InpFldNum, Token
			);
#endif

		}

		InpFldNum++;
	} // while

	// loop over data fields looking for unset fields
	for(
		FldNum = 0;
		FldNum < pDb->DataFldCnt;
		FldNum++
	) {
		if (!pFieldSet[FldNum]) {
			DefaultData(pDb->pDataFlds[FldNum], &pDataArray[FldNum]);
		}
	} // for each data field

	return TRUE;

}

void CheckCategories(
	FIELD_DATA	*pFieldData,
	CATEGORY	**ppCatRec
) {
	char CatFld[MAX_CAT_FIELD+1];
	// +3 for null and surounding semicolons
	char OneCat[MAX_CAT_FIELD+3];
	char CatRec[MAX_CAT_RECORD+3] = ";";
	char *p;

	if (strlen(pFieldData->pStr) > MAX_CAT_FIELD) {
		fprintf(
			stderr,
			"Warning: category too long: "
			"%s ignored.\n", 
			pFieldData->pStr
		);
	} else {

		strcpy(CatFld, pFieldData->pStr);

		strcat(CatRec, (*ppCatRec)->CategoryString);
		strcat(CatRec, ";");
		StrLower(CatRec);

#if DEBUG
		printf("Rec: %s\n", CatRec);
		printf("Fld: %s\n", CatFld);
#endif
		// Find first category
    	p = strtok(CatFld, ";" );
    	while( p != NULL ) {

			strcpy(OneCat, ";");
			strcat(OneCat, p);
			strcat(OneCat, ";");
			StrLower(OneCat);

#if DEBUG
			printf("Cat: %s\n", OneCat);
#endif
			if (!strstr(CatRec, OneCat)) {

				// list of all cats can not exceed MAX_CAT_RECORD chars.
				if (
					// + 1 for added ";"
					strlen(p) + 1 +
					strlen((*ppCatRec)->CategoryString)	<=
					MAX_CAT_RECORD
				) {

					(*ppCatRec)->RecHdr.Length +=
						strlen(p) + 1;

					if (!(
						*ppCatRec = realloc(
							*ppCatRec, (*ppCatRec)->RecHdr.Length)
					))
						Fatal("Out of memory adding category");

					if (strlen((*ppCatRec)->CategoryString))
						strcat((*ppCatRec)->CategoryString, ";");

					strcat( (*ppCatRec)->CategoryString, p);

#if DEBUG
					printf("New categories: %s\n",
						(*ppCatRec)->CategoryString
					);
#endif
				} else {
					fprintf(
						stderr,
						"Warning: too many new categories, "
						"%s ignored.\n", 
						p
					);
				}
			} // if new category
#if DEBUG
			else {
				printf("already present: %s\n", OneCat);
			}
#endif
        	p = strtok( NULL, ";" );   // Find next category
		} // while
	}

}

void WriteRecord(
	GDB			*pDb,
	FILE		*OutFile,
	unsigned	RecNo,
	FIELD_DATA	*pDataArray,
	TYPE_INDEX	*pTypeIndexArray,
	CATEGORY	**ppCatRec
) {

// The initial size of the data record
#define INITIAL_SIZE 200

	int field;
	int FreeOffset=pDb->FixedSize;
	FIELDDEF *pFld;
	FIELD_DATA	*pFieldData;
	DATABYTERECORD *pRec;
	unsigned *pWord;
	int DataSize;

	DataSize = max(INITIAL_SIZE, pDb->FixedSize);
	if(!( pRec = malloc(sizeof(pRec->RecHdr) + DataSize)))
		Fatal("Out of memory for initial data record");

	memset(pRec, 0, sizeof(pRec->RecHdr) + DataSize);

	// Process each data field
	for (
		field=0, pFld = pDb->pDataFlds[0], pFieldData = pDataArray;
		field < pDb->DataFldCnt;
		field++, pFld = pDb->pDataFlds[field], pFieldData++
	) {
		pWord = (unsigned *)(pRec->Fixed + pFld->DataOffset);

		if (
			(pFld->Flags & FIELDDEF_RELATIVE)	&& 
			pFld->FieldType != STRING_FIELD		&&
			pFld->FieldType != COMBO_FIELD		&&
			pFld->FieldType != PHONE_FIELD		&&
			pFld->FieldType != NUMBER_FIELD		&&
			pFld->FieldType != CURRENCY_FIELD	&&
			pFld->FieldType != CATEGORY_FIELD	&&
			pFld->FieldType != MULTILINE_FIELD	&&
			pFld->FieldType != USER_FIELD
		)
			Fatal("relative non-string");

		switch (pFld->FieldType) {
			case NOTE_FIELD:
			case RADIO_FIELD:
			case TIME_FIELD:
				*pWord = pFieldData->word;
				break;

			case BYTEBOOL_FIELD:
				// Drop the flag
				pRec->Fixed[pFld->DataOffset] &= ~pFld->Reserved;
				// Raise the flag if imported data is "true"
				if (pFieldData->boolean)
					pRec->Fixed[pFld->DataOffset] |= pFld->Reserved;
				break;

			case WORDBOOL_FIELD:
				// Drop the flag
				*pWord &= ~pFld->Reserved;
				// Raise the flag if imported data is "true"
				if (pFieldData->boolean)
					*pWord |= pFld->Reserved;
				break;
	 			pDb->FixedSize += 2;
				break;

	 		case DATE_FIELD:
				memcpy(
					pWord,
					pFieldData->date,
					sizeof(pFieldData->date)
				);
				break;

			case STRING_FIELD:
			case COMBO_FIELD:
			case PHONE_FIELD:
			case NUMBER_FIELD:
			case CURRENCY_FIELD:
			case CATEGORY_FIELD:
			case MULTILINE_FIELD:
			case USER_FIELD:
				if (!(pFld->Flags & FIELDDEF_RELATIVE))
					Fatal("non-relative string");

				// null ptr or zero-length string ?
				if (
					pFieldData->pStr == NULL ||
					strlen(pFieldData->pStr) == 0
				) {
					// Point fixed offset to shared null at end of fixed data
					*pWord = pDb->FixedSize-1;
				} else {

					if (pFld->FieldType == CATEGORY_FIELD)
						CheckCategories(pFieldData, ppCatRec);

					if ( FreeOffset + strlen(pFieldData->pStr) + 1 > DataSize) {
						// grow the record
						DataSize = FreeOffset + strlen(pFieldData->pStr) + 1;
						if(!(pRec =
							realloc(pRec, sizeof(pRec->RecHdr) + DataSize)))
								Fatal("Out of memory growing data record");
						pWord = (unsigned *)(pRec->Fixed + pFld->DataOffset);
					}
					*pWord = FreeOffset;
					memcpy(
						pRec->Fixed + FreeOffset,
						pFieldData->pStr,
						strlen(pFieldData->pStr) + 1
					);
					free(pFieldData->pStr);
					FreeOffset += strlen(pFieldData->pStr) + 1;
				}

				break;
		} // switch
	} // for

	pRec->RecHdr.Type = TYPE_DATA;
	pRec->RecHdr.Status = MODIFIED;
	pRec->RecHdr.Length = sizeof(pRec->RecHdr) + FreeOffset;
	pRec->RecHdr.Number = RecNo;

#if DEBUG
	DumpBytes((unsigned char *)pRec, pRec->RecHdr.Length, 0);
#endif

	AddToIndex(&pRec->RecHdr, ftell(OutFile), pTypeIndexArray);

	if(fwrite(pRec, pRec->RecHdr.Length, 1, OutFile) != 1)
		Fatal("Write of new record failed");

	free(pRec);
}

void CopyGDB(
	GDB *pDb,
	FILE *OutGDB,
	TYPE_INDEX *pTypeIndexArray,
	CATEGORY **ppCatRec
) {

	DBHEADER		NewHdr;
	int				NumRecords;
	RECORDHEADER	*pRec;


	*ppCatRec = NULL;

	if(fwrite(HPSIGNATURE, sizeof(SIGNATURE), 1, OutGDB) != 1)
		Fatal("Write of new database signature failed");

	NewHdr = pDb->Hdr;
	NewHdr.LookupSeek = 0; // mark Lookup and TypeFirst tables as missing

	AddToIndex(&NewHdr.RecHdr, ftell(OutGDB), pTypeIndexArray);
	if(fwrite(&NewHdr, sizeof(NewHdr), 1, OutGDB) != 1)
		Fatal("Write of new database header failed(1)");

	NumRecords = NewHdr.NumRecords - 2;

#if DEBUG
	printf("Copying %d records\n", NumRecords);
#endif

	fseek(pDb->f, sizeof(SIGNATURE)+sizeof(DBHEADER), SEEK_SET);
	while (NumRecords--) {
		pRec = NULL;
		pRec = ReadRec(pDb, pRec, 0);

		if (pRec) {
#if DEBUG
printf("Copy record, type = %d, number = %d\n", pRec->Type, pRec->Number);
#endif

			// Sometimes the NumRecords field in the database header is
			// too high. Does it include deleted records? Anyway stop if
			// we reach the lookup table.
			if (pRec->Type == TYPE_LOOKUPTABLE)
				break;

			if (pRec->Type == TYPE_CATEGORY)
				*ppCatRec = (CATEGORY *)pRec;
			else {
				AddToIndex(pRec, ftell(OutGDB), pTypeIndexArray);
				if(fwrite(pRec, pRec->Length, 1, OutGDB) != 1)
					Fatal("Write of new database record failed");

				free(pRec);
			}
		}
	} // while
#if DEBUG
	printf("Copied %d records\n", NewHdr.NumRecords - 2 - NumRecords);
#endif

}

void ImportData(GDB *pDb, FILE *InCDF, FILE *OutGDB) {

	DBHEADER	NewHdr;
	unsigned	DataRecNo;
	unsigned	InpRecNum=1;
	LOOKUPTABLE	*pLuTab;
	TYPEFIRST   Tf;
	FIELD_DATA	*pDataArray;
	TYPE_INDEX	TypeIndexArray[32] = {0};
	CATEGORY	*pCatRec;

	CopyGDB(pDb, OutGDB, TypeIndexArray, &pCatRec);

	DataRecNo = pDb->Tf[TYPE_DATA+1] - pDb->Tf[TYPE_DATA];
#if DEBUG
	printf("Adding to %d data records\n", DataRecNo);
#endif

	if(!(pDataArray = calloc(pDb->DataFldCnt, sizeof(FIELD_DATA)))) {
		Fatal("Out of memory for field data array");
	}

	NewHdr = pDb->Hdr;
	NewHdr.ViewptHash = 0; // force rebuild of viewpoint tables

	while (1) {

		if (!ImportRecord(pDb, InCDF, InpRecNum++, pDataArray))
			break;

		WriteRecord(
			pDb,
			OutGDB,
			DataRecNo++,
			pDataArray,
			TypeIndexArray,
			&pCatRec
		);

		NewHdr.NumRecords++;

	}

	if (pCatRec) {
		AddToIndex(&pCatRec->RecHdr, ftell(OutGDB), TypeIndexArray);
		if(fwrite(pCatRec, pCatRec->RecHdr.Length, 1, OutGDB) != 1)
			Fatal("Write of new category record failed");

		free(pCatRec);
	}

	BuildIndex(&pLuTab, Tf, TypeIndexArray);

#if INDEX
	if (IndexOpt) {
	
		NewHdr.LookupSeek =	ftell(OutGDB);

		memcpy(
			pLuTab->Lookup[Tf[TYPE_LOOKUPTABLE]].Offset,
			&NewHdr.LookupSeek,
			3
		);

		if(fwrite(pLuTab, pLuTab->RecHdr.Length, 1, OutGDB) != 1)
			Fatal("Write of new database lookup table failed");

		if(fwrite(Tf, sizeof(Tf), 1, OutGDB) != 1)
			Fatal("Write of new database type-first table failed");
	} else {
		NewHdr.LookupSeek =	0;
	}
#else
	NewHdr.LookupSeek =	0;
#endif

	free(pLuTab);

#if DEBUG
	printf("new DB has %d data records\n", DataRecNo);
#endif

	fseek(OutGDB, sizeof(SIGNATURE), SEEK_SET);
	if(fwrite(&NewHdr, sizeof(NewHdr), 1, OutGDB) != 1)
		Fatal("Write of new database header failed(2)");

	free(pDataArray);

}


GDB *OpenGdb(char *Name)
/*
 * Open database given by name, returns pointer to allocated DB descriptor.
 * (GDB structure)
 *
 */
{
	GDB *pDb;
	CATEGORY *pCats;
	int Size;
	char *p;

	/* allocate memory for GDB structure */
	if(!(pDb = malloc(sizeof(GDB)))) Fatal("Out of mem for GDB");

	/* Try to open file */
	if(!(pDb->f = fopen(Name, "rb"))) Fatal("Open file failed for %s", Name);

	CheckSignature(pDb);
	GetDbHeader(pDb);
	GetIndex(pDb);
	GetFields(pDb);

	if (StructOpt) {
		Seek2Record(pDb, TYPE_CATEGORY, 0, &Size);
		if(Size) {
			pCats = ReadRec(pDb, NULL, Size);
			if(pCats && pCats->RecHdr.Type == TYPE_CATEGORY) {
				
				printf("\nCategories:\n");

				// Find first category
    			p = strtok( pCats->CategoryString, ";" );
    			while( p != NULL ) {
        			printf( "  %s\n", p );
        			p = strtok( NULL, ";" );   // Find next category
    			}
			}
		}
	}

	return pDb;
}


void CloseGdb(GDB *pDb)
/*
 * Close open GDB, close associated file, free associated memory.
 * GDB *pDb: pointer to DB descriptor.
 *
 */
{
	fclose(pDb->f);			// close file
	free(pDb->pLuTab);		// free lookup table record
	free(pDb->pFld);		// free fielddef array
	free(pDb);				// free descriptor itself
}


void Usage(void)
/*
 * Output some usage info 
 *
 */
{
	printf("Usage: GDBIO <in-GDB> [in-CDF out-GDB]\n\n");

	printf("Version 0.95: Imports and exports comma-delimited ASCII files to and\n");
	printf("from existing HP 100LX database (*.?DB) files.  If one file name is\n");
	printf("given then GDBIO can display its structure or export its contents. If\n");
	printf("three files are given then GDBIO will combine the comma-delimited\n");
	printf("data in the second with the GDB-formatted data in the first and\n");
	printf("create the third in GDB format. The new database is not valid until\n");
	printf("opened by the 100LX database application which rebuilds the file's\n");
	printf("internal index.  The import and export formats are the same allowing\n");
	printf("round trips. Notes can not be imported - yet. The double-quotes\n");
	printf("surrounding fields are optional.  Options can be added to any\n");
	printf("argument after a slash ('/').  Valid option letters are:\n\n");

	printf("       opt:           Meaning:\n");
	printf("       ----  ----------------------------------\n");
	printf("        S     show structure of database\n");
	printf("        N     list contents of notes (implies X)\n");
#if ASCII_OPT
	printf("        A     disable LICS <> Extended ASCII mapping\n");
#endif
	printf("        X     export data records, comma delimited\n");
	printf("              (default when only in-GDB provided)\n\n");

	printf("Internet: 72672,3706@compuserve.com, CIS: 72672,3706\n");

}


/* --------------------------------------------------------------------
								main
	-------------------------------------------------------------------- */
int main(int argc, char *argv[])
{
	GDB *pDb;
	FILE *InCDF;
	FILE *OutGDB;
	int arg;
	char *pSlash;
	BOOLEAN NoOptLetters=TRUE;

#if DEBUG && ASCII_OPT
	{
		int i;

		for (i = 32; i < 256; i++)
			if (
				(ToASCII[ToLICS[i]] != (unsigned char) i) &&
				(ToLICS[i] != NoLICS) &&
				(ToASCII[ToLICS[i]] != NoASCII)
			)
				printf("%c (%d) not mapped properly\n", i, i);
	}
#endif

	if(argc < 2) {
		Usage();
		return EXIT_FAILURE;
	}

	ExportOpt = FALSE;
	NotesOpt = FALSE;
	StructOpt = FALSE;
#if ASCII_OPT
	ExtAscOpt = FALSE;
#endif
	ImportOpt = FALSE;
#if INDEX
	IndexOpt = FALSE;
#endif

	// scan and remove options
	for (arg = 1; arg < argc; arg++) {
		while (pSlash = strrchr(argv[arg], '/')) {
			switch (tolower(*(pSlash+1))) {

				case 's':
					StructOpt = TRUE;
					NoOptLetters=FALSE;
					break;

				case 'n':
					NotesOpt = TRUE;
					NoOptLetters=FALSE;
					break;

				case 'x':
					ExportOpt = TRUE;
					NoOptLetters=FALSE;
					break;

#if ASCII_OPT
				case 'a':
					ExtAscOpt = TRUE;
					NoOptLetters=FALSE;
					break;
#endif

#if INDEX
				case 'i':
					IndexOpt = TRUE;
					NoOptLetters=FALSE;
					break;

#endif
				// Unknown option
				default:
					Usage();
					Fatal("Unknown option letter");

			}
			*pSlash = 0;	// remove option
		}

		if (strlen(argv[arg]) == 0)
			continue;

	}

	if (NoOptLetters)
		// export is the default if only input-gdb is provided
		if (argc == 2) ExportOpt = TRUE;

	pDb = OpenGdb(argv[1]);

	if (argc == 4 && strlen(argv[2]) && strlen(argv[3])) {
		ImportOpt = TRUE;
		if(!(InCDF = fopen(argv[2], "r")))
			Fatal("Open file failed for %s", argv[2]);
		if(!(OutGDB = fopen(argv[3], "wb")))
			Fatal("Open file failed for %s", argv[3]);
	}

	if (NotesOpt) ExportOpt = TRUE;	/* N opt implies D opt */
	if (ExportOpt) OutputData(pDb);
	if (ImportOpt) ImportData(pDb, InCDF, OutGDB);
	CloseGdb(pDb);
	if (ImportOpt) {
		fclose(InCDF);
		if (fclose(OutGDB))
			Fatal("error closing output file");
	}

	return EXIT_SUCCESS;
}

